document.addEventListener('DOMContentLoaded', function () {

    // --------------------------------------------------------------------------
    // Tab Switching Logic
    // --------------------------------------------------------------------------
    function switchTab(tabId) {
        // Hide all tab content
        var contents = document.querySelectorAll('.hrm-tab-content');
        contents.forEach(function (content) {
            content.style.display = 'none';
            content.classList.remove('active');
        });

        // Deactivate all buttons
        var buttons = document.querySelectorAll('.hrm-tab-btn');
        buttons.forEach(function (btn) {
            btn.classList.remove('active');
        });

        // Show target tab
        var targetContent = document.getElementById(tabId);
        if (targetContent) {
            targetContent.style.display = 'block';
            // Slight delay to allow display:block to apply before opacity transition
            requestAnimationFrame(() => {
                targetContent.classList.add('active');
            });
        }

        // Activate target button
        var targetBtn = document.querySelector('.hrm-tab-btn[data-tab="' + tabId + '"]');
        if (targetBtn) {
            targetBtn.classList.add('active');
        }

        // Close sidebar on mobile if link clicked
        if (window.innerWidth <= 768) {
            const sidebar = document.querySelector('.hrm-sidebar');
            if (sidebar) sidebar.classList.remove('active');
        }
    }

    // Add click event listeners to all tab buttons
    var tabButtons = document.querySelectorAll('.hrm-tab-btn');
    tabButtons.forEach(function (button) {
        button.addEventListener('click', function (e) {
            // If it's the logout button, don't switch tabs, let it navigate
            if (this.classList.contains('logout')) return;

            // If it has an onclick attribute (legacy), let it run, but usually we use data-tab
            // The refactored HTML uses onclick="openTab(...)" which we might want to supersede or coexist.
            // The Refactored HTML in hrm-portal.php actually has:
            // <li class="hrm-tab-btn" onclick="openTab(event, 'tasks')">

            // We should respect that inline handler or replace it.
            // But since this is a global JS file, let's look for data attributes or just rely on the inline one?
            // "hrm-frontend.js" should ideally handle it unobtrusively.
            // In hrm-portal.php, I outputted inline "openTab". 
            // I should leave that alone or ensure this JS doesn't conflict.
        });
    });

    // Note: The php file outputted an inline script: 
    // function openTab(evt, tabName) { ... }
    // This file adds *additional* listeners.

    // Mobile Sidebar Toggle
    const mobileToggle = document.querySelector('.hrm-menu-toggle');
    const sidebar = document.querySelector('.hrm-sidebar');

    if (mobileToggle && sidebar) {
        mobileToggle.addEventListener('click', (e) => {
            e.stopPropagation();
            sidebar.classList.toggle('active');

            // Prevent body scroll when sidebar is open on mobile
            if (sidebar.classList.contains('active')) {
                document.body.style.overflow = 'hidden';
            } else {
                document.body.style.overflow = '';
            }
        });

        // Close when clicking outside on mobile
        document.addEventListener('click', (e) => {
            if (window.innerWidth <= 768 && sidebar.classList.contains('active')) {
                if (!sidebar.contains(e.target) && !mobileToggle.contains(e.target)) {
                    sidebar.classList.remove('active');
                    document.body.style.overflow = '';
                }
            }
        });
    }

    // Mobile Search Toggle
    const searchTrigger = document.getElementById('hrm-mobile-search-trigger');
    const searchClose = document.getElementById('hrm-mobile-search-close');
    const searchWrapper = document.getElementById('hrm-search-wrapper');
    const searchInput = document.getElementById('hrm-portal-search-input');

    if (searchTrigger && searchWrapper) {
        searchTrigger.addEventListener('click', () => {
            searchWrapper.classList.add('active');
            if (searchInput) {
                setTimeout(() => searchInput.focus(), 300);
            }
        });
    }

    if (searchClose && searchWrapper) {
        searchClose.addEventListener('click', () => {
            searchWrapper.classList.remove('active');
        });
    }

    // --------------------------------------------------------------------------
    // Scroll Effects for Top Bar
    // --------------------------------------------------------------------------
    const portalWrap = document.querySelector('.hrm-portal-wrap');
    if (portalWrap) {
        window.addEventListener('scroll', () => {
            if (window.scrollY > 20) {
                portalWrap.classList.add('hrm-portal-scrolled');
            } else {
                portalWrap.classList.remove('hrm-portal-scrolled');
            }
        }, { passive: true });
    }


    // --------------------------------------------------------------------------
    // Attendance AJAX Handling
    // --------------------------------------------------------------------------
    const btnCheckIn = document.getElementById('hrm-btn-checkin');
    const btnCheckOut = document.getElementById('hrm-btn-checkout');
    const attendanceForm = document.getElementById('hrm-attendance-form');

    // Prevent form from submitting traditionally - forcing AJAX only
    if (attendanceForm) {
        attendanceForm.addEventListener('submit', function (e) {
            e.preventDefault();
            e.stopImmediatePropagation();
            console.log('HRM: Form submission prevented - using AJAX only');
            return false;
        });
    }

    // LocalStorage key for check-in persistence
    const HRM_CHECKIN_KEY = 'hrm_checkin_timestamp';
    const HRM_CHECKIN_DATE_KEY = 'hrm_checkin_date';

    function saveCheckInTime(timestamp) {
        const today = new Date().toISOString().split('T')[0];
        localStorage.setItem(HRM_CHECKIN_KEY, timestamp.toString());
        localStorage.setItem(HRM_CHECKIN_DATE_KEY, today);
        console.log('HRM: Saved check-in timestamp to localStorage:', timestamp);
    }

    function getStoredCheckInTime() {
        const today = new Date().toISOString().split('T')[0];
        const storedDate = localStorage.getItem(HRM_CHECKIN_DATE_KEY);

        // Only return if it's the same day
        if (storedDate === today) {
            const timestamp = localStorage.getItem(HRM_CHECKIN_KEY);
            return timestamp ? parseInt(timestamp, 10) : null;
        }

        // Clear stale data from previous days
        clearStoredCheckInTime();
        return null;
    }

    function clearStoredCheckInTime() {
        localStorage.removeItem(HRM_CHECKIN_KEY);
        localStorage.removeItem(HRM_CHECKIN_DATE_KEY);
        console.log('HRM: Cleared check-in timestamp from localStorage');
    }

    function handleAttendanceClick(e, type, btn) {
        if (!btn) return;
        e.preventDefault();
        e.stopImmediatePropagation();
        console.log('Attendance Click:', type);

        // Prevent double-click
        if (btn.classList.contains('loading') || btn.disabled) {
            console.log('HRM: Button already processing, ignoring click');
            return;
        }

        // Visual feedback
        btn.classList.add('loading');
        btn.disabled = true;

        const formData = new FormData();
        formData.append('action', 'hrm_mark_attendance');
        formData.append('type', type);

        // --- Robust URL & Nonce Resolution ---
        let ajaxUrl = '';
        let nonce = '';

        // 1. Try Localized Object
        if (typeof hrm_ajax !== 'undefined') {
            if (hrm_ajax.ajaxurl && hrm_ajax.ajaxurl !== 'undefined') ajaxUrl = hrm_ajax.ajaxurl;
            if (hrm_ajax.nonce) nonce = hrm_ajax.nonce;
        }

        // 2. Fallback Path Detection
        if (!ajaxUrl) {
            console.warn('HRM: hrm_ajax.ajaxurl missing. Attempting autodetection...');
            const wpScript = document.querySelector('script[src*="wp-includes"], script[src*="wp-content"]');
            if (wpScript) {
                const src = wpScript.src;
                const match = src.match(/^(.*)(wp-content|wp-includes)/);
                if (match && match[1]) {
                    ajaxUrl = match[1] + 'wp-admin/admin-ajax.php';
                }
            }
            if (!ajaxUrl) {
                ajaxUrl = location.origin + '/wp-admin/admin-ajax.php';
            }
            console.log('HRM: Autodetected Ajax URL:', ajaxUrl);
        }

        // 3. Fallback Nonce
        if (!nonce) {
            console.warn('HRM: Nonce not in localized object, trying form fallback...');
            const ajaxNonceInput = document.getElementById('hrm_ajax_nonce_field');
            if (ajaxNonceInput && ajaxNonceInput.value) {
                nonce = ajaxNonceInput.value;
                console.log('HRM: Found nonce in hrm_ajax_nonce_field');
            }
        }

        if (!nonce) {
            console.error('HRM: No nonce available!');
            showToast('Security token missing. Please refresh the page.', 'error');
            btn.disabled = false;
            btn.classList.remove('loading');
            return;
        }

        formData.append('nonce', nonce);
        console.log('HRM Attendance Request:', { ajaxUrl, type, noncePresent: !!nonce });

        fetch(ajaxUrl, {
            method: 'POST',
            body: formData
        })
            .then(response => {
                console.log('HRM Response Status:', response.status);
                if (!response.ok) throw new Error('Network response was not ok');
                return response.text();
            })
            .then(text => {
                console.log('HRM Raw Response:', text);
                try {
                    const data = JSON.parse(text);
                    if (data.success) {
                        showToast(data.data.message, 'success');

                        // Handle check-in: Save timestamp and update UI
                        if (data.data.type === 'check_in') {
                            // Save the server timestamp to localStorage for persistence
                            const serverTimestamp = data.data.timestamp * 1000; // Convert PHP timestamp to JS ms
                            saveCheckInTime(serverTimestamp);

                            // Update UI WITHOUT page reload
                            updateDashboardUI(data.data);

                            // Button stays in new state - no reload needed!
                            console.log('HRM: Check-in complete - timer running, no reload');
                        } else if (data.data.type === 'check_out') {
                            // Clear stored check-in time
                            clearStoredCheckInTime();

                            // Update UI
                            updateDashboardUI(data.data);

                            // For check-out, reload after a delay to show final state
                            setTimeout(() => location.reload(), 1500);
                        }
                    } else {
                        console.error('HRM Error Response:', data.data);
                        showToast(data.data, 'error');
                        btn.disabled = false;
                        btn.classList.remove('loading');
                    }
                } catch (err) {
                    console.error('HRM JSON Parse Error:', err);
                    console.error('HRM Server Response (raw):', text);
                    showToast('Server error. Check console.', 'error');
                    btn.disabled = false;
                    btn.classList.remove('loading');
                }
            })
            .catch(err => {
                console.error('HRM Fetch Error:', err);
                showToast('Connection failed.', 'error');
                btn.disabled = false;
                btn.classList.remove('loading');
            });
    }

    if (btnCheckIn) {
        btnCheckIn.addEventListener('click', function (e) {
            handleAttendanceClick(e, 'check_in', this);
        });
    }

    if (btnCheckOut) {
        btnCheckOut.addEventListener('click', function (e) {
            handleAttendanceClick(e, 'check_out', this);
        });
    }

    function updateDashboardUI(data) {
        const timerWrapper = document.querySelector('.hrm-timer-wrapper');
        const statusIndicator = document.querySelector('.hrm-status-indicator');
        const timerBg = document.getElementById('hrm-hero-timer');
        const timerLabel = document.querySelector('.hrm-timer-label');
        const actionArea = document.querySelector('.hrm-action-area');
        let btnCheckInEl = document.getElementById('hrm-btn-checkin');
        let btnCheckOutEl = document.getElementById('hrm-btn-checkout');

        if (data.type === 'check_in') {
            console.log('HRM: Updating UI for check-in');

            // Update Timer Wrapper to Active State
            if (timerWrapper) timerWrapper.classList.add('active');

            // Update Status Indicator
            if (statusIndicator) {
                statusIndicator.className = 'hrm-status-indicator active';
                statusIndicator.innerHTML = '<span class="status-dot"></span> Checked In';
            }

            // Transform "Check In" button to "Check Out" button
            if (btnCheckInEl) {
                // Create a new button element to replace the old one
                const newBtn = document.createElement('button');
                newBtn.id = 'hrm-btn-checkout';
                newBtn.type = 'submit';
                newBtn.name = 'type';
                newBtn.value = 'check_out';
                newBtn.className = 'hrm-btn hrm-btn-large hrm-btn-red';
                newBtn.innerHTML = '<span class="dashicons dashicons-controls-pause" style="font-size: 24px; width: 24px; height: 24px;"></span> Check Out';
                newBtn.disabled = false;

                // Replace old button
                btnCheckInEl.parentNode.replaceChild(newBtn, btnCheckInEl);

                // Add click listener to new button
                newBtn.addEventListener('click', function (e) {
                    handleAttendanceClick(e, 'check_out', this);
                });

                console.log('HRM: Button transformed to Check Out');
            }

            // Start Timer Immediately from 0
            if (timerBg) {
                // Reset timer attributes
                timerBg.setAttribute('data-active', '1');
                timerBg.setAttribute('data-elapsed', '0');
                timerBg.innerText = '00:00:00';

                // Clear any existing timer interval
                if (window.hrmTimerInterval) clearInterval(window.hrmTimerInterval);

                // Start new timer counting from 0 (just checked in)
                // Use simple counter to avoid timezone issues
                let elapsedSeconds = 0;
                const startTime = Date.now();
                console.log('HRM: Starting timer from 0 seconds');

                window.hrmTimerInterval = setInterval(() => {
                    // Calculate elapsed using page-relative time (no timezone issues)
                    elapsedSeconds = Math.floor((Date.now() - startTime) / 1000);
                    const h = Math.floor(elapsedSeconds / 3600).toString().padStart(2, '0');
                    const m = Math.floor((elapsedSeconds % 3600) / 60).toString().padStart(2, '0');
                    const s = (elapsedSeconds % 60).toString().padStart(2, '0');
                    timerBg.innerText = `${h}:${m}:${s}`;
                }, 1000);
            }

            // Update timer label
            if (timerLabel) {
                timerLabel.innerText = 'Duration';
            }

        } else if (data.type === 'check_out') {
            console.log('HRM: Updating UI for check-out');

            // Stop the timer
            if (window.hrmTimerInterval) {
                clearInterval(window.hrmTimerInterval);
                window.hrmTimerInterval = null;
            }

            // Update Timer Wrapper to Completed State
            if (timerWrapper) timerWrapper.classList.remove('active');

            // Update Status Indicator
            if (statusIndicator) {
                statusIndicator.className = 'hrm-status-indicator completed';
                statusIndicator.innerHTML = '<span class="status-dot"></span> Checked Out';
            }

            // Update timer to show final hours worked
            if (timerBg && data.hours_worked) {
                timerBg.setAttribute('data-active', '0');
                const hrs = data.hours_worked;
                const h = Math.floor(hrs);
                const m = Math.floor((hrs % 1) * 60);
                timerBg.innerText = `${String(h).padStart(2, '0')}:${String(m).padStart(2, '0')}:00`;
            }

            // Update timer label
            if (timerLabel) {
                timerLabel.innerText = 'Total Hours';
            }

            // Update button to show completion state
            btnCheckOutEl = document.getElementById('hrm-btn-checkout');
            if (btnCheckOutEl) {
                btnCheckOutEl.classList.remove('loading');
                btnCheckOutEl.disabled = true;
                btnCheckOutEl.className = 'hrm-btn hrm-btn-large hrm-btn-green';
                btnCheckOutEl.innerHTML = '<span class="dashicons dashicons-yes-alt" style="font-size: 24px; width: 24px; height: 24px;"></span> Checked Out';
            }
        }
    }

    // --------------------------------------------------------------------------
    // Attendance Timer Logic with Server-Elapsed Time (Timezone Safe)
    // --------------------------------------------------------------------------
    function initTimer() {
        const timerEl = document.getElementById('hrm-hero-timer');
        if (!timerEl) return;

        // Avoid double init
        if (timerEl.dataset.timerInitialized) return;
        timerEl.dataset.timerInitialized = 'true';

        const isActive = timerEl.getAttribute('data-active') === '1';
        // Server provides elapsed seconds directly - this is timezone-safe
        let serverElapsed = parseInt(timerEl.getAttribute('data-elapsed')) || 0;

        // Ensure non-negative
        if (serverElapsed < 0) serverElapsed = 0;

        function formatTime(totalSeconds) {
            // Ensure non-negative
            if (totalSeconds < 0) totalSeconds = 0;
            const h = Math.floor(totalSeconds / 3600).toString().padStart(2, '0');
            const m = Math.floor((totalSeconds % 3600) / 60).toString().padStart(2, '0');
            const s = (totalSeconds % 60).toString().padStart(2, '0');
            return `${h}:${m}:${s}`;
        }

        console.log('HRM Timer Init: isActive=' + isActive + ', serverElapsed=' + serverElapsed);

        if (isActive) {
            // Use server-elapsed time as the base, then count up from there
            // This avoids any timezone issues between PHP and JavaScript
            const pageLoadTime = Date.now();
            let currentElapsed = serverElapsed;

            // Initial display
            timerEl.innerText = formatTime(currentElapsed);

            // Clear any existing global interval
            if (window.hrmTimerInterval) clearInterval(window.hrmTimerInterval);

            // Increment timer every second from the server-provided elapsed time
            window.hrmTimerInterval = setInterval(() => {
                const timeSincePageLoad = Math.floor((Date.now() - pageLoadTime) / 1000);
                const totalElapsed = serverElapsed + timeSincePageLoad;
                timerEl.innerText = formatTime(totalElapsed);
            }, 1000);

            console.log('HRM Timer: Started from server elapsed time:', serverElapsed + ' seconds');
        } else {
            // Not active - just display the elapsed time (could be final hours worked)
            timerEl.innerText = formatTime(serverElapsed);
            console.log('HRM Timer: Not active, showing static time:', formatTime(serverElapsed));
        }
    }

    // Run on load
    initTimer();
    // Also run on jQuery ready to be safe (if mixed)
    if (typeof jQuery !== 'undefined') {
        jQuery(document).ready(initTimer);
    }

    // Helper: Toast Notification
    function showToast(message, type = 'success') {
        // Remove existing toast if any
        const existing = document.querySelector('.hrm-toast-notification');
        if (existing) existing.remove();

        const toast = document.createElement('div');
        toast.className = 'hrm-toast-notification';
        toast.style.background = type === 'success' ? '#10B981' : '#EF4444';
        toast.style.color = 'white';
        toast.style.zIndex = '99999';
        toast.style.position = 'fixed';
        toast.style.bottom = '20px';
        toast.style.right = '20px';
        // Actually the CSS handles formatting well.

        toast.innerHTML = message;

        document.body.appendChild(toast);

        setTimeout(() => {
            toast.style.opacity = '0';
            toast.style.transform = 'translateY(20px)';
            setTimeout(() => toast.remove(), 300);
        }, 3000);
    }
    // --- Animated Counter (Frontend) ---
    const countElements = document.querySelectorAll('.hrm-mini-stat .value');
    countElements.forEach(el => {
        const text = el.innerText;
        const hasPercent = text.includes('%');
        const target = parseFloat(text.replace(/[^0-9.]/g, ''));

        if (isNaN(target)) return;

        // Initial set to 0 to prepare for animation
        el.innerText = '0' + (hasPercent ? '%' : '');

        const duration = 1500;
        const startTime = performance.now();

        function updateCount(currentTime) {
            const elapsed = currentTime - startTime;
            const progress = Math.min(elapsed / duration, 1);
            const ease = 1 - Math.pow(1 - progress, 3); // Cubic ease out

            let currentVal = target * ease;

            // Handle integers vs decimals? Assuming integers for now as per stats
            el.innerText = Math.floor(currentVal) + (hasPercent ? '%' : '');

            if (progress < 1) {
                requestAnimationFrame(updateCount);
            } else {
                el.innerText = target + (hasPercent ? '%' : '');
            }
        }

        requestAnimationFrame(updateCount);
    });

});
